<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="<?php echo e(csrf_token()); ?>">
    <title>Chat with Report</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
</head>

<body class="bg-slate-50">
<div class="min-h-screen flex flex-col">

    <!-- Header -->
    <div class="bg-white border-b border-slate-200 px-6 py-4 shadow-sm">
        <div class="flex items-center justify-between max-w-4xl mx-auto w-full">
            <div>
                <h1 class="text-2xl font-bold text-slate-900">Report Analysis</h1>
                <p class="text-sm text-slate-600 mt-1">Upload and analyze medical reports</p>
            </div>
            <div class="flex items-center gap-4">
                <button id="shareBtn" class="text-slate-600 hover:text-blue-600 text-sm font-medium flex items-center gap-1 hidden">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.368 2.684 3 3 0 00-5.368-2.684z"></path></svg>
                    Share
                </button>
                <a href="<?php echo e(url('/')); ?>" class="text-slate-600 hover:text-slate-900 text-sm font-medium">Back</a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="flex-1 px-6 py-6 max-w-4xl mx-auto w-full">

        <!-- Dropzone -->
        <div class="mb-8">
            <div id="dropzone"
                 class="border-2 border-dashed border-slate-300 rounded-xl p-8 text-center cursor-pointer
                        hover:border-blue-500 hover:bg-blue-50 transition-all">

                <svg class="w-12 h-12 mx-auto text-slate-400 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                          d="M12 4v16m8-8H4"></path>
                </svg>

                <p class="text-slate-600 font-medium">Drop medical report here or click to upload</p>
                <p class="text-slate-500 text-sm mt-1">Supports PDF or images</p>

                <input type="file" id="fileInput" accept="image/*,.pdf" class="hidden">
            </div>
        </div>

        <!-- Analysis Results -->
        <div style="display: none;" id="resultsContainer" class="hidden mb-8 space-y-4">
            <div style="display: none;" class="bg-white rounded-xl border border-slate-200 p-6">
                <h3 class="font-bold text-slate-900 mb-3">Analysis Results</h3>
                <div class="bg-slate-50 rounded-lg p-4 overflow-auto max-h-40">
                    <pre id="jsonContent" class="text-xs text-slate-700"></pre>
                </div>
            </div>

            <div class="bg-white rounded-xl border border-slate-200 p-6">
                <h3 class="font-bold text-slate-900 mb-3">AI Explanation</h3>
                <p id="aiExplanation" class="text-slate-700 leading-relaxed"></p>
            </div>
        </div>

        <!-- Chat -->
        <div class="bg-white rounded-xl border border-slate-200 flex flex-col" style="height: 500px;">

            <div id="chatContainer" class="flex-1 overflow-y-auto p-6 space-y-4">
                <div class="bg-slate-200 text-slate-900 rounded-2xl rounded-bl-sm px-4 py-3 max-w-md">
                    <p class="text-sm">Upload a report to start chatting.</p>
                </div>
            </div>

            <div class="border-t border-slate-200 p-4 flex gap-3">
                <textarea id="messageInput"
                          placeholder="Ask about the report..."
                          class="flex-1 px-4 py-3 border border-slate-300 rounded-lg
                                 focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                          rows="2"></textarea>

                <button id="sendBtn"
                        class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-medium">
                    Send
                </button>
            </div>
        </div>

    </div>
</div>

<script>
// Blade-provided endpoints and auth info
const startUrl = "<?php echo e(route('reports.start')); ?>";
const uploadUrl = "<?php echo e(route('reports.uploadLabImage')); ?>";
const chatUrl = "<?php echo e(route('reports.chat')); ?>";
const historyUrl = "<?php echo e(route('reports.history', ':id')); ?>"; // Placeholder for ID
const AUTH_USER_ID = <?php echo e(json_encode(auth()->id())); ?>;

// Check if we are resuming an existing report
const existingReport = <?php echo json_encode($report ?? null, 15, 512) ?>;

// Configure axios to include Laravel CSRF token
if (document.querySelector('meta[name="csrf-token"]')) {
    axios.defaults.headers.common['X-CSRF-TOKEN'] = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
}

    const dropzone = document.getElementById('dropzone');
    const fileInput = document.getElementById('fileInput');
    const resultsContainer = document.getElementById('resultsContainer');
    const jsonContent = document.getElementById('jsonContent');
    const aiExplanation = document.getElementById('aiExplanation');
    const chatContainer = document.getElementById('chatContainer');
    const messageInput = document.getElementById('messageInput');
    const sendBtn = document.getElementById('sendBtn');
    const shareBtn = document.getElementById('shareBtn');

    let currentReportData = null;

    // Initialize if existing report
    if (existingReport) {
        currentReportData = { report_id: existingReport.id };
        shareBtn.classList.remove('hidden'); // Show share button
        loadHistory(existingReport.id);
    }

    shareBtn.onclick = async () => {
        if (!currentReportData?.report_id) return;
        try {
            const response = await axios.post(`/share/generate/report/${currentReportData.report_id}`, {}, {
                headers: { 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') }
            });
            if (response.data.status === 'success') {
                navigator.clipboard.writeText(response.data.url);
                alert('Link copied to clipboard!');
            }
        } catch (error) {
            console.error('Error generating link:', error);
            const msg = error.response?.data?.message || 'Failed to generate share link.';
            alert(msg);
        }
    };

    async function loadHistory(reportId) {
        try {
            const url = historyUrl.replace(':id', reportId);
            const res = await axios.get(url);
            if (res.data.status === 'success') {
                chatContainer.innerHTML = ''; // Clear initial message
                res.data.history.forEach(msg => {
                    appendMessage(msg.message, 'user');
                    if (msg.response_en || msg.response_ar) {
                        appendAiMessage(msg.response_en, msg.response_ar);
                    }
                });
            }
        } catch (err) {
            console.error('Failed to load history', err);
        }
    }

    // Dropzone events
    dropzone.onclick = () => fileInput.click();

    dropzone.ondragover = e => {
        e.preventDefault();
        dropzone.classList.add('border-blue-500', 'bg-blue-50');
    };

    dropzone.ondragleave = () => {
        dropzone.classList.remove('border-blue-500', 'bg-blue-50');
    };

    dropzone.ondrop = e => {
        e.preventDefault();
        dropzone.classList.remove('border-blue-500', 'bg-blue-50');
        handleFile(e.dataTransfer.files[0]);
    };

    fileInput.onchange = e => handleFile(e.target.files[0]);

    async function handleFile(file) {
        if (!file) return;

        // Ensure we have a report_id: create one if needed
        if (!currentReportData || !currentReportData.report_id) {
            try {
                const startRes = await axios.post(startUrl, {
                    user_id: AUTH_USER_ID || 1,
                    title: `Report - ${new Date().toLocaleString()}`
                });
                currentReportData = { report_id: startRes.data.report_id };
            } catch (err) {
                console.error('Failed to start report', err);
                appendMessage('Failed to start report.', 'ai');
                return;
            }
        }

        const formData = new FormData();
        formData.append('image', file);
        formData.append('report_id', currentReportData.report_id);
        // formData.append('user_id', AUTH_USER_ID || 1);

        try {
            const res = await axios.post(uploadUrl, formData, { headers: { 'Content-Type': 'multipart/form-data' } });
            // upload endpoint returns OCR info; now ask the backend for an initial analysis
            // We'll request a concise analysis via the chat endpoint
            try {
                const chatRes = await axios.post(chatUrl, {
                    message: 'Please analyze the uploaded report and provide a concise, plain-language summary in English.',
                    report_id: currentReportData.report_id
                    // user_id: AUTH_USER_ID || 1
                });

                const chatData = chatRes.data;
                // Normalize into a displayable object
                const display = {
                    analysis: res.data.ocr_text ? { ocr_text: res.data.ocr_text } : null,
                    explanation: chatData.chat?.response_en || chatData.response_en || chatData.response || null,
                    report_id: currentReportData.report_id
                };

                currentReportData = Object.assign(currentReportData, display);
                shareBtn.classList.remove('hidden'); // Show share button
                displayResults(display);
                // show the model's analysis in both English and Arabic when available
                const aiEn = chatData.chat?.response_en || chatData.response_en || chatData.response || null;
                const aiAr = chatData.chat?.response_ar || chatData.response_ar || null;
                if (aiEn || aiAr) {
                    appendAiMessage(aiEn, aiAr);
                } else {
                    appendMessage('Report analyzed successfully.', 'ai');
                }
            } catch (err2) {
                console.error('Chat analysis failed', err2);
                appendMessage('Upload succeeded but analysis failed.', 'ai');
            }

        } catch (e) {
            console.error(e);
            appendMessage('Upload failed.', 'ai');
        }
    }

    function displayResults(data) {
        resultsContainer.classList.remove('hidden');
        jsonContent.textContent = JSON.stringify(data.analysis, null, 2);
        aiExplanation.textContent = data.explanation || formatReportAnalysis(data.analysis);
    }

    function formatReportAnalysis(analysis) {
        if (!analysis) return 'No analysis available.';
        if (analysis.ocr_text) {
            return analysis.ocr_text.trim();
        }
        return 'Analysis complete.';
    }

    sendBtn.onclick = sendMessage;
    messageInput.onkeypress = e => {
        if (e.key === "Enter" && !e.shiftKey) {
            e.preventDefault();
            sendMessage();
        }
    };

    async function sendMessage() {
        const msg = messageInput.value.trim();
        if (!msg || !currentReportData?.report_id) return;

        appendMessage(msg, "user");
        messageInput.value = "";

        try {
            const res = await axios.post(chatUrl, {
                message: msg,
                report_id: currentReportData.report_id,
                // user_id: AUTH_USER_ID || 1
            });

            const aiEn = res.data.chat?.response_en || res.data.response_en || res.data.response || res.data.reply || null;
            const aiAr = res.data.chat?.response_ar || res.data.response_ar || null;
            if (aiEn || aiAr) {
                appendAiMessage(aiEn, aiAr);
            } else {
                appendMessage('No reply from server.', 'ai');
            }
        } catch (err) {
            console.error(err);
            appendMessage("Error occurred.", "ai");
        }
    }

    function appendMessage(text, type) {
        const wrap = document.createElement('div');
        wrap.className = `flex items-end gap-3 ${type === "user" ? "justify-end" : ""}`;

        const bubble = document.createElement('div');
        bubble.className = type === "user"
            ? "bg-blue-600 text-white rounded-2xl rounded-br-sm px-4 py-3 max-w-md"
            : "bg-slate-200 text-slate-900 rounded-2xl rounded-bl-sm px-4 py-3 max-w-md";
        bubble.innerHTML = `<p class="text-sm">${escapeHtml(text)}</p>`;
        wrap.appendChild(bubble);

        chatContainer.appendChild(wrap);
        chatContainer.scrollTop = chatContainer.scrollHeight;
    }

    function appendAiMessage(en, ar) {
        const wrap = document.createElement('div');
        wrap.className = 'flex items-start gap-3';

        const bubble = document.createElement('div');
        bubble.className = 'bg-slate-200 text-slate-900 rounded-2xl rounded-bl-sm px-4 py-3 max-w-2xl';

        let inner = '';
        if (en) {
            inner += `<p class="text-sm font-medium">${escapeHtml(en)}</p>`;
        }
        if (ar) {
            inner += `<p class="text-sm text-slate-600 mt-2">${escapeHtml(ar)}</p>`;
        }

        if (!inner) inner = `<p class="text-sm">No reply from server.</p>`;

        bubble.innerHTML = inner;
        wrap.appendChild(bubble);

        chatContainer.appendChild(wrap);
        chatContainer.scrollTop = chatContainer.scrollHeight;
    }

    function escapeHtml(text){
        if (!text && text !== 0) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
</script>

</body>
</html>
<?php /**PATH D:\Tammeny\resources\views/chat-report.blade.php ENDPATH**/ ?>