
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="<?php echo e(csrf_token()); ?>">
<title>Chat with X-Ray</title>
<script src="https://cdn.tailwindcss.com"></script>
<script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
</head>
<body class="bg-slate-50">

<div class="min-h-screen flex flex-col">
    <!-- Header -->
    <div class="bg-white border-b border-slate-200 px-6 py-4 shadow-sm">
        <div class="flex items-center justify-between max-w-4xl mx-auto w-full">
            <div>
                <h1 class="text-2xl font-bold text-slate-900">X-Ray Analysis</h1>
                <p class="text-sm text-slate-600 mt-1">Upload and analyze medical X-rays</p>
            </div>
            <div class="flex items-center gap-4">
                <button id="shareBtn" class="text-slate-600 hover:text-blue-600 text-sm font-medium flex items-center gap-1 hidden">
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.368 2.684 3 3 0 00-5.368-2.684z"></path></svg>
                    Share
                </button>
                <a href="<?php echo e(url('/')); ?>" class="text-slate-600 hover:text-slate-900 text-sm font-medium">Back</a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="flex-1 px-6 py-6 max-w-4xl mx-auto w-full">
        <!-- Upload Section -->
        <div class="mb-8">
            <div id="dropzone" class="border-2 border-dashed border-slate-300 rounded-xl p-8 text-center cursor-pointer hover:border-blue-500 hover:bg-blue-50 transition-all">
                <svg class="w-12 h-12 mx-auto text-slate-400 mb-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                </svg>
                <p class="text-slate-600 font-medium">Drop X-ray image here or click to upload</p>
                <p class="text-slate-500 text-sm mt-1">Supports JPG, PNG, and other image formats</p>
                <input type="file" id="fileInput" accept="image/*" class="hidden" />
            </div>
        </div>

        <!-- Analysis Results -->
        <div id="resultsContainer" class="hidden mb-8 space-y-4">
            <div style="display: none;" class="bg-white rounded-xl border border-slate-200 p-6">
                <h3 class="font-bold text-slate-900 mb-3">Analysis Results</h3>
                <div id="jsonResult" class="bg-slate-50 rounded-lg p-4 overflow-auto max-h-40">
                    <pre id="jsonContent" class="text-xs text-slate-700"></pre>
                </div>
            </div>

            <div style="display: none;" class="bg-white rounded-xl border border-slate-200 p-6">
                <h3 class="font-bold text-slate-900 mb-3">AI Analysis</h3>
                <p id="aiExplanation" class="text-slate-700 leading-relaxed"></p>
            </div>
        </div>

        <!-- Chat Area -->
        <div class="bg-white rounded-xl border border-slate-200 flex flex-col" style="height: 500px;">
            <div id="chatContainer" class="flex-1 overflow-y-auto p-6 space-y-4">
                <div class="flex items-end gap-3">
                    <div class="bg-slate-200 text-slate-900 rounded-2xl rounded-bl-sm px-4 py-3 max-w-md">
                        <p class="text-sm">Upload an X-ray image to get started with analysis and ask questions.</p>
                    </div>
                </div>
            </div>

            <div class="border-t border-slate-200 p-4 flex gap-3">
                <textarea id="messageInput" placeholder="Ask about the X-ray analysis..." class="flex-1 px-4 py-3 border border-slate-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none" rows="2"></textarea>
                <button id="sendBtn" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-medium transition-colors h-fit">Send</button>
            </div>
        </div>
    </div>
</div>

<script>
// Blade-provided endpoints and auth info
const uploadUrl = "<?php echo e(route('radiology.upload')); ?>";
const chatUrl = "<?php echo e(route('radiology.chat')); ?>";
const historyUrl = "<?php echo e(route('radiology.history', ':id')); ?>";
const AUTH_USER_ID = <?php echo e(json_encode(auth()->id())); ?>;

// Check if we are resuming an existing report
const existingReport = <?php echo json_encode($report ?? null, 15, 512) ?>;

// Configure axios to include Laravel CSRF token
if (document.querySelector('meta[name="csrf-token"]')) {
    axios.defaults.headers.common['X-CSRF-TOKEN'] = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
}

const dropzone = document.getElementById('dropzone');
const fileInput = document.getElementById('fileInput');
const chatContainer = document.getElementById('chatContainer');
const messageInput = document.getElementById('messageInput');
const sendBtn = document.getElementById('sendBtn');
const resultsContainer = document.getElementById('resultsContainer');
const jsonContent = document.getElementById('jsonContent');
const aiExplanation = document.getElementById('aiExplanation');
const shareBtn = document.getElementById('shareBtn');

let currentXrayData = null; // هنا هنخزن الريسبونس كامل بعد الرفع

// Initialize if existing report
if (existingReport) {
    currentXrayData = { report_id: existingReport.id };
    shareBtn.classList.remove('hidden'); // Show share button
    // Optionally display existing analysis if stored in DB (it is in raw_json)
    if (existingReport.raw_json) {
        try {
            const analysis = JSON.parse(existingReport.raw_json);
            displayResults({
                analysis: analysis,
                explanation: existingReport.impression
            });
        } catch(e) { console.error('Error parsing existing JSON', e); }
    }
    loadHistory(existingReport.id);
}

shareBtn.onclick = async () => {
    if (!currentXrayData?.report_id) return;
    try {
        const response = await axios.post(`/share/generate/radiology/${currentXrayData.report_id}`, {}, {
            headers: { 'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content') }
        });
        if (response.data.status === 'success') {
            navigator.clipboard.writeText(response.data.url);
            alert('Link copied to clipboard!');
        }
    } catch (error) {
        console.error('Error generating link:', error);
        const msg = error.response?.data?.message || 'Failed to generate share link.';
        alert(msg);
    }
};

async function loadHistory(reportId) {
    try {
        const url = historyUrl.replace(':id', reportId);
        const res = await axios.get(url);
        if (res.data.status === 'success') {
             chatContainer.innerHTML = ''; // Clear initial message
             res.data.history.forEach(msg => {
                appendMessage(msg.message, 'user');
                if (msg.response_en || msg.response_ar) {
                    // Xray chat history structure might be slightly different, let's check controller
                    // RadiologyController history returns RadiologyMessage model
                    // RadiologyMessage has response_ar, response_en
                    // appendMessage handles single content, we might need a helper for dual language or just append twice
                    // The existing appendMessage takes content and type.
                    // Let's just append both for now or create a better bubble.
                    // Actually, let's use the same logic as chat-report if possible, but here appendMessage is simpler.
                    // Let's just append EN then AR.
                    if(msg.response_en) appendMessage(msg.response_en, 'ai');
                    if(msg.response_ar) appendMessage(msg.response_ar, 'ai');
                }
            });
        }
    } catch (err) {
        console.error('Failed to load history', err);
    }
}

// Dropzone Events
dropzone.addEventListener('click', () => fileInput.click());
dropzone.addEventListener('dragover', e => { e.preventDefault(); dropzone.classList.add('border-blue-500', 'bg-blue-50'); });
dropzone.addEventListener('dragleave', () => { dropzone.classList.remove('border-blue-500', 'bg-blue-50'); });
dropzone.addEventListener('drop', e => {
    e.preventDefault();
    dropzone.classList.remove('border-blue-500', 'bg-blue-50');
    handleFile(e.dataTransfer.files[0]);
});
fileInput.addEventListener('change', e => handleFile(e.target.files[0]));

// Upload File Function
async function handleFile(file) {
    if (!file) return;

    const formData = new FormData();
    formData.append('image', file);
    // formData.append('user_id', AUTH_USER_ID || 1); // use authenticated id when available

    try {
        const response = await axios.post(uploadUrl, formData, { headers: { 'Content-Type': 'multipart/form-data' } });
        currentXrayData = response.data;
        shareBtn.classList.remove('hidden'); // Show share button
        displayResults(response.data);
        appendMessage("X-ray uploaded and analyzed successfully.", 'ai');
    } catch (error) {
        console.error(error);
        appendMessage("Error uploading X-ray image.", 'ai');
    }
}

// Display Analysis
function displayResults(data) {
    jsonContent.textContent = JSON.stringify(data.analysis, null, 2);
    // If the backend provided a plain explanation, prefer it; otherwise generate a clear summary
    const summary = data.explanation || formatAnalysis(data.analysis);
    aiExplanation.textContent = summary;
    resultsContainer.classList.remove('hidden');
}

function formatAnalysis(analysis){
    if(!analysis) return 'No analysis available.';

    const parts = [];

    if(analysis.image_summary){
        parts.push(analysis.image_summary);
    }

    if(Array.isArray(analysis.observations) && analysis.observations.length){
        parts.push('\nFindings:');
        analysis.observations.forEach((obs, i) => {
            const region = obs.region ? `${obs.region}` : '';
            const type = obs.type ? ` (${obs.type.replace(/_/g, ' ')})` : '';
            const confidence = (typeof obs.confidence === 'number') ? ` Confidence: ${Math.round(obs.confidence * 100)}%.` : '';
            parts.push(`- ${obs.description}${region ? ' — ' + region : ''}${type}${confidence}`);
        });
    }

    if(analysis.flags){
        if(analysis.flags.abnormality_present){
            parts.push('\nAssessment: An abnormality is present.');
            if(analysis.flags.urgent_attention_suggested){
                parts.push('This finding may require urgent medical attention. Please consult a clinician promptly.');
            } else {
                parts.push('Clinical correlation is recommended.');
            }
        } else {
            parts.push('\nAssessment: No significant abnormality detected.');
        }
    }

    return parts.join('\n');
}

// Chat
sendBtn.addEventListener('click', sendMessage);
messageInput.addEventListener('keypress', e => { if(e.key === 'Enter' && !e.shiftKey){ e.preventDefault(); sendMessage(); } });

async function sendMessage() {
    const message = messageInput.value.trim();
    if (!message || !currentXrayData?.report_id) return;

    appendMessage(message, 'user');
    messageInput.value = '';

    try {
        const response = await axios.post(chatUrl, {
            message: message,
            report_id: currentXrayData.report_id,
            // user_id: AUTH_USER_ID || 1
        });

        appendMessage(response.data.response_ar, 'ai');
        appendMessage(response.data.response_en, 'ai');
    } catch (error) {
        console.error(error);
        appendMessage("Error processing your message.", 'ai');
    }
}

// Append Chat Bubble
function appendMessage(content, type) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `flex items-end gap-3 ${type === 'user' ? 'justify-end' : ''}`;
    const bubble = document.createElement('div');
    bubble.className = type === 'user'
        ? 'bg-blue-600 text-white rounded-2xl rounded-br-sm px-4 py-3 max-w-md'
        : 'bg-slate-200 text-slate-900 rounded-2xl rounded-bl-sm px-4 py-3 max-w-md';
    bubble.innerHTML = `<p class="text-sm">${escapeHtml(content)}</p>`;
    messageDiv.appendChild(bubble);
    chatContainer.appendChild(messageDiv);
    chatContainer.scrollTop = chatContainer.scrollHeight;
}

function escapeHtml(text){
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
</script>
</body>
</html>
<?php /**PATH D:\Tammeny\resources\views/chat-xray.blade.php ENDPATH**/ ?>