<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class FileHelper
{
    /**
     * Get the configured upload disk name
     */
    public static function getUploadDisk(): string
    {
        return env('UPLOAD_DISK', 'public');
    }

    /**
     * Get the configured upload path within the disk
     */
    public static function getUploadPath(): string
    {
        return env('UPLOAD_PATH', 'uploads');
    }

    /**
     * Check if disk is cloud-based (S3, etc.)
     */
    private static function isCloudDisk(string $disk): bool
    {
        $driver = config("filesystems.disks.{$disk}.driver");
        return in_array($driver, ['s3', 'ftp', 'sftp']);
    }

    /**
     * Store an uploaded file and return both the relative path and full system path
     * 
     * For S3: uploads to S3, downloads to temp for AI processing
     * For local: uses direct path
     * 
     * @param UploadedFile $file The uploaded file
     * @param string|null $subdirectory Optional subdirectory within the upload path
     * @return array ['relative_path' => string, 'full_path' => string, 'is_temp' => bool]
     */
    public static function storeUploadedFile(UploadedFile $file, ?string $subdirectory = null): array
    {
        $disk = self::getUploadDisk();
        $basePath = self::getUploadPath();
        
        // Build the storage path
        $storagePath = $subdirectory ? "{$basePath}/{$subdirectory}" : $basePath;
        
        // Generate unique filename
        $fileName = uniqid() . '.' . $file->getClientOriginalExtension();
        
        // Store the file
        $relativePath = $file->storeAs($storagePath, $fileName, $disk);
        
        // Get the full system path
        if (self::isCloudDisk($disk)) {
            // For S3: download to temp for AI processing
            $tempPath = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $fileName;
            $content = Storage::disk($disk)->get($relativePath);
            file_put_contents($tempPath, $content);
            
            return [
                'relative_path' => $relativePath,
                'full_path' => $tempPath,
                'filename' => $fileName,
                'is_temp' => true  // Flag to clean up later
            ];
        }
        
        // For local storage
        $fullPath = Storage::disk($disk)->path($relativePath);
        
        return [
            'relative_path' => $relativePath,
            'full_path' => $fullPath,
            'filename' => $fileName,
            'is_temp' => false
        ];
    }

    /**
     * Clean up temporary file if created for cloud storage
     */
    public static function cleanupTempFile(array $fileData): void
    {
        if (($fileData['is_temp'] ?? false) && file_exists($fileData['full_path'])) {
            @unlink($fileData['full_path']);
        }
    }

    /**
     * Get the public URL for a stored file
     * 
     * @param string $relativePath The relative path stored in the database
     * @return string The public URL
     */
    public static function getFileUrl(string $relativePath): string
    {
        $disk = self::getUploadDisk();
        return Storage::disk($disk)->url($relativePath);
    }

    /**
     * Get the full system path for a stored file
     * 
     * @param string $relativePath The relative path stored in the database
     * @return string The full system path
     */
    public static function getFilePath(string $relativePath): string
    {
        $disk = self::getUploadDisk();
        return Storage::disk($disk)->path($relativePath);
    }

    /**
     * Delete a stored file
     * 
     * @param string $relativePath The relative path stored in the database
     * @return bool
     */
    public static function deleteFile(string $relativePath): bool
    {
        $disk = self::getUploadDisk();
        return Storage::disk($disk)->delete($relativePath);
    }
}